<?php
/**
 * Plugin Name: RSYS CDN
 * Plugin URI: https://rsys.cloud/wordpress
 * Description: Official RSYS Cloud CDN integration for WordPress. Accelerate your website with global CDN.
 * Version: 2.2.0
 * Author: RSYS Cloud
 * Author URI: https://rsys.cloud
 * License: GPL v2 or later
 * Text Domain: rsys-cdn
 */

if (!defined('ABSPATH')) exit;

define('RSYS_CDN_VERSION', '2.2.0');
define('RSYS_SITE_URL', 'https://rsys.cloud');

class RSYS_CDN {
    private static $instance = null;
    private $options;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->options = get_option('rsys_cdn_options', array(
            'enabled' => false,
            'cdn_url' => '',
            'include_images' => true,
            'include_css' => true,
            'include_js' => true,
            'exclude_paths' => ''
        ));
        
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_ajax_rsys_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_rsys_test_connection', array($this, 'ajax_test_connection'));
        
        if ($this->options['enabled'] && !empty($this->options['cdn_url'])) {
            add_action('template_redirect', array($this, 'start_buffer'));
            add_action('shutdown', array($this, 'end_buffer'));
        }
    }
    
    public function add_admin_menu() {
        add_menu_page('RSYS CDN', 'RSYS CDN', 'manage_options', 'rsys-cdn', array($this, 'render_admin_page'), 'dashicons-performance', 80);
    }
    
    public function enqueue_admin_assets($hook) {
        if ($hook !== 'toplevel_page_rsys-cdn') return;
        remove_all_actions('admin_notices');
        remove_all_actions('all_admin_notices');
    }
    
    public function render_admin_page() {
        if (!current_user_can('manage_options')) return;
        ?>
        <!DOCTYPE html>
        <html>
        <head>
            <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
            <style>
                #wpwrap { background: #030014 !important; }
                #wpcontent, #wpbody-content { padding: 0 !important; }
                .rsys-app * { box-sizing: border-box; margin: 0; padding: 0; }
                .rsys-app {
                    font-family: 'Inter', -apple-system, sans-serif;
                    background: linear-gradient(135deg, #030014 0%, #0a0a1f 50%, #030014 100%);
                    min-height: 100vh;
                    color: #fff;
                    position: relative;
                    overflow: hidden;
                }
                
                /* Stars */
                .rsys-stars {
                    position: fixed;
                    top: 0; left: 0; right: 0; bottom: 0;
                    background-image: 
                        radial-gradient(1px 1px at 10% 20%, rgba(255,255,255,0.8), transparent),
                        radial-gradient(1px 1px at 20% 50%, rgba(255,255,255,0.6), transparent),
                        radial-gradient(1px 1px at 30% 30%, rgba(255,255,255,0.7), transparent),
                        radial-gradient(1px 1px at 40% 70%, rgba(255,255,255,0.5), transparent),
                        radial-gradient(1px 1px at 50% 40%, rgba(255,255,255,0.8), transparent),
                        radial-gradient(1px 1px at 60% 80%, rgba(255,255,255,0.6), transparent),
                        radial-gradient(1px 1px at 70% 10%, rgba(255,255,255,0.7), transparent),
                        radial-gradient(1px 1px at 80% 60%, rgba(255,255,255,0.5), transparent),
                        radial-gradient(1px 1px at 90% 90%, rgba(255,255,255,0.8), transparent),
                        radial-gradient(2px 2px at 15% 85%, rgba(255,255,255,0.4), transparent),
                        radial-gradient(2px 2px at 85% 15%, rgba(255,255,255,0.4), transparent);
                    animation: twinkle 4s ease-in-out infinite alternate;
                    pointer-events: none;
                    z-index: 0;
                }
                @keyframes twinkle { 0% { opacity: 0.7; } 100% { opacity: 1; } }
                
                /* Nebula */
                .rsys-nebula {
                    position: fixed;
                    border-radius: 50%;
                    filter: blur(80px);
                    opacity: 0.12;
                    pointer-events: none;
                    z-index: 0;
                }
                .rsys-nebula-1 {
                    width: 500px; height: 500px;
                    top: -150px; right: -100px;
                    background: radial-gradient(circle, #6366f1 0%, #8b5cf6 40%, transparent 70%);
                }
                .rsys-nebula-2 {
                    width: 400px; height: 400px;
                    bottom: -100px; left: -100px;
                    background: radial-gradient(circle, #00d4ff 0%, #0ea5e9 40%, transparent 70%);
                }
                
                /* Planets */
                .rsys-planet {
                    position: fixed;
                    border-radius: 50%;
                    pointer-events: none;
                    z-index: 1;
                }
                .rsys-planet-1 {
                    width: 120px; height: 120px;
                    top: 10%; right: 8%;
                    background: linear-gradient(135deg, #1e1b4b 0%, #312e81 50%, #4338ca 100%);
                    box-shadow: 
                        inset -20px -20px 40px rgba(0,0,0,0.6),
                        inset 10px 10px 20px rgba(99,102,241,0.3),
                        0 0 60px rgba(99,102,241,0.2);
                    animation: planetFloat1 25s ease-in-out infinite;
                }
                .rsys-planet-1::after {
                    content: '';
                    position: absolute;
                    top: 50%; left: -30%;
                    width: 160%; height: 20px;
                    background: linear-gradient(90deg, transparent, rgba(139,92,246,0.4), transparent);
                    border-radius: 50%;
                    transform: translateY(-50%) rotateX(75deg);
                }
                .rsys-planet-2 {
                    width: 60px; height: 60px;
                    bottom: 15%; left: 5%;
                    background: linear-gradient(135deg, #0c4a6e 0%, #0369a1 50%, #0ea5e9 100%);
                    box-shadow: 
                        inset -10px -10px 20px rgba(0,0,0,0.5),
                        0 0 40px rgba(14,165,233,0.3);
                    animation: planetFloat2 18s ease-in-out infinite;
                }
                .rsys-planet-3 {
                    width: 35px; height: 35px;
                    top: 60%; right: 15%;
                    background: linear-gradient(135deg, #581c87 0%, #7c3aed 100%);
                    box-shadow: 0 0 25px rgba(124,58,237,0.4);
                    animation: planetFloat3 15s ease-in-out infinite;
                }
                @keyframes planetFloat1 {
                    0%, 100% { transform: translateY(0) rotate(0deg); }
                    50% { transform: translateY(-20px) rotate(5deg); }
                }
                @keyframes planetFloat2 {
                    0%, 100% { transform: translateY(0) translateX(0); }
                    50% { transform: translateY(15px) translateX(10px); }
                }
                @keyframes planetFloat3 {
                    0%, 100% { transform: translateY(0); }
                    50% { transform: translateY(-10px); }
                }
                
                /* Container */
                .rsys-container {
                    position: relative;
                    z-index: 10;
                    max-width: 800px;
                    margin: 0 auto;
                    padding: 40px 20px;
                }
                
                /* Header */
                .rsys-header {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    margin-bottom: 30px;
                    padding: 20px 25px;
                    background: rgba(15, 15, 35, 0.6);
                    border: 1px solid rgba(255,255,255,0.08);
                    border-radius: 16px;
                    backdrop-filter: blur(20px);
                }
                .rsys-logo {
                    display: flex;
                    align-items: center;
                    gap: 14px;
                }
                .rsys-logo-icon {
                    width: 44px; height: 44px;
                    background: linear-gradient(135deg, #6366f1, #8b5cf6);
                    border-radius: 12px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    box-shadow: 0 8px 25px rgba(99,102,241,0.35);
                }
                .rsys-logo-icon svg { width: 24px; height: 24px; }
                .rsys-logo-text {
                    font-size: 1.5rem;
                    font-weight: 700;
                    color: #fff;
                }
                .rsys-logo-text span { color: #00d4ff; }
                .rsys-version {
                    padding: 6px 14px;
                    background: rgba(99,102,241,0.15);
                    border: 1px solid rgba(99,102,241,0.25);
                    border-radius: 20px;
                    font-size: 0.8rem;
                    color: #a5b4fc;
                    font-weight: 500;
                }
                
                /* Status */
                .rsys-status {
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    padding: 16px 20px;
                    margin-bottom: 25px;
                    border-radius: 12px;
                    border: 1px solid;
                }
                .rsys-status.active {
                    background: rgba(16, 185, 129, 0.08);
                    border-color: rgba(16, 185, 129, 0.25);
                }
                .rsys-status.inactive {
                    background: rgba(239, 68, 68, 0.08);
                    border-color: rgba(239, 68, 68, 0.25);
                }
                .rsys-status-icon {
                    width: 36px; height: 36px;
                    border-radius: 10px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                .rsys-status.active .rsys-status-icon { background: rgba(16, 185, 129, 0.2); }
                .rsys-status.inactive .rsys-status-icon { background: rgba(239, 68, 68, 0.2); }
                .rsys-status-icon svg { width: 20px; height: 20px; }
                .rsys-status.active .rsys-status-icon svg { color: #10b981; }
                .rsys-status.inactive .rsys-status-icon svg { color: #ef4444; }
                .rsys-status-text { font-size: 0.95rem; font-weight: 500; }
                .rsys-status.active .rsys-status-text { color: #10b981; }
                .rsys-status.inactive .rsys-status-text { color: #ef4444; }
                
                /* Card */
                .rsys-card {
                    background: rgba(15, 15, 35, 0.6);
                    border: 1px solid rgba(255,255,255,0.08);
                    border-radius: 16px;
                    backdrop-filter: blur(20px);
                    margin-bottom: 20px;
                    overflow: hidden;
                }
                .rsys-card-header {
                    padding: 18px 22px;
                    border-bottom: 1px solid rgba(255,255,255,0.06);
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }
                .rsys-card-header svg { width: 20px; height: 20px; color: #6366f1; }
                .rsys-card-header h2 { font-size: 1.05rem; font-weight: 600; color: #fff; }
                .rsys-card-body { padding: 22px; }
                
                /* Form */
                .rsys-form-group { margin-bottom: 22px; }
                .rsys-form-group > label {
                    display: block;
                    font-size: 0.85rem;
                    font-weight: 500;
                    color: #9ca3af;
                    margin-bottom: 8px;
                }
                .rsys-input {
                    width: 100%;
                    padding: 12px 16px;
                    background: rgba(0,0,0,0.25);
                    border: 1px solid rgba(255,255,255,0.1);
                    border-radius: 10px;
                    color: #fff;
                    font-size: 0.95rem;
                    font-family: inherit;
                    transition: all 0.2s;
                }
                .rsys-input:focus {
                    outline: none;
                    border-color: #6366f1;
                    box-shadow: 0 0 0 3px rgba(99,102,241,0.15);
                }
                .rsys-input::placeholder { color: #4b5563; }
                .rsys-help { font-size: 0.78rem; color: #6b7280; margin-top: 6px; }
                
                /* Toggle */
                .rsys-toggle {
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    cursor: pointer;
                }
                .rsys-toggle input { display: none; }
                .rsys-toggle-switch {
                    width: 48px; height: 26px;
                    background: rgba(255,255,255,0.1);
                    border-radius: 26px;
                    position: relative;
                    transition: all 0.3s;
                }
                .rsys-toggle-switch::after {
                    content: '';
                    position: absolute;
                    width: 20px; height: 20px;
                    background: #fff;
                    border-radius: 50%;
                    top: 3px; left: 3px;
                    transition: all 0.3s;
                }
                .rsys-toggle input:checked + .rsys-toggle-switch {
                    background: linear-gradient(135deg, #6366f1, #8b5cf6);
                }
                .rsys-toggle input:checked + .rsys-toggle-switch::after { left: 25px; }
                .rsys-toggle-label { font-size: 0.95rem; font-weight: 500; color: #fff; }
                
                /* Checkbox Grid */
                .rsys-checkbox-grid {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 12px;
                }
                .rsys-checkbox-item {
                    display: flex;
                    align-items: center;
                    gap: 10px;
                    padding: 14px;
                    background: rgba(0,0,0,0.2);
                    border: 1px solid rgba(255,255,255,0.08);
                    border-radius: 10px;
                    cursor: pointer;
                    transition: all 0.2s;
                }
                .rsys-checkbox-item:hover { border-color: rgba(99,102,241,0.4); }
                .rsys-checkbox-item input { display: none; }
                .rsys-checkbox-box {
                    width: 20px; height: 20px;
                    border: 2px solid rgba(255,255,255,0.25);
                    border-radius: 5px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: all 0.2s;
                    flex-shrink: 0;
                }
                .rsys-checkbox-item input:checked + .rsys-checkbox-box {
                    background: linear-gradient(135deg, #6366f1, #8b5cf6);
                    border-color: transparent;
                }
                .rsys-checkbox-item input:checked + .rsys-checkbox-box svg { display: block; }
                .rsys-checkbox-box svg { display: none; width: 12px; height: 12px; color: #fff; }
                .rsys-checkbox-label { font-size: 0.9rem; color: #e5e7eb; display: flex; align-items: center; gap: 6px; }
                .rsys-checkbox-label svg { width: 16px; height: 16px; color: #9ca3af; }
                
                /* Buttons */
                .rsys-btn {
                    display: inline-flex;
                    align-items: center;
                    gap: 8px;
                    padding: 12px 22px;
                    font-size: 0.9rem;
                    font-weight: 600;
                    font-family: inherit;
                    border: none;
                    border-radius: 10px;
                    cursor: pointer;
                    transition: all 0.2s;
                }
                .rsys-btn svg { width: 18px; height: 18px; }
                .rsys-btn-primary {
                    background: linear-gradient(135deg, #6366f1, #8b5cf6);
                    color: #fff;
                    box-shadow: 0 6px 20px rgba(99,102,241,0.3);
                }
                .rsys-btn-primary:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 8px 25px rgba(99,102,241,0.4);
                }
                .rsys-btn-secondary {
                    background: rgba(255,255,255,0.08);
                    color: #e5e7eb;
                    border: 1px solid rgba(255,255,255,0.12);
                }
                .rsys-btn-secondary:hover { background: rgba(255,255,255,0.12); }
                .rsys-btn-group { display: flex; gap: 12px; margin-top: 8px; }
                
                /* Info */
                .rsys-info {
                    display: flex;
                    align-items: flex-start;
                    gap: 12px;
                    padding: 16px;
                    background: rgba(0,212,255,0.06);
                    border: 1px solid rgba(0,212,255,0.15);
                    border-radius: 10px;
                    margin-top: 20px;
                }
                .rsys-info svg { width: 20px; height: 20px; color: #00d4ff; flex-shrink: 0; margin-top: 2px; }
                .rsys-info-text { color: #9ca3af; font-size: 0.85rem; line-height: 1.5; }
                .rsys-info-text a { color: #00d4ff; text-decoration: none; }
                .rsys-info-text a:hover { text-decoration: underline; }
                .rsys-info-text strong { color: #e5e7eb; }
                
                /* Footer */
                .rsys-footer { text-align: center; padding: 25px; color: #6b7280; font-size: 0.8rem; }
                .rsys-footer a { color: #6366f1; text-decoration: none; }
                
                /* Toast */
                .rsys-toast {
                    position: fixed;
                    bottom: 25px; right: 25px;
                    padding: 14px 20px;
                    border-radius: 10px;
                    font-size: 0.9rem;
                    font-weight: 500;
                    z-index: 9999;
                    transform: translateX(150%);
                    transition: transform 0.3s ease;
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }
                .rsys-toast.show { transform: translateX(0); }
                .rsys-toast.success { background: #10b981; color: #fff; }
                .rsys-toast.error { background: #ef4444; color: #fff; }
                .rsys-toast svg { width: 18px; height: 18px; }
                
                /* Loading */
                .rsys-loading {
                    width: 18px; height: 18px;
                    border: 2px solid rgba(255,255,255,0.3);
                    border-top-color: #fff;
                    border-radius: 50%;
                    animation: spin 0.7s linear infinite;
                }
                @keyframes spin { to { transform: rotate(360deg); } }
            </style>
        </head>
        <body>
            <div class="rsys-app">
                <div class="rsys-stars"></div>
                <div class="rsys-nebula rsys-nebula-1"></div>
                <div class="rsys-nebula rsys-nebula-2"></div>
                <div class="rsys-planet rsys-planet-1"></div>
                <div class="rsys-planet rsys-planet-2"></div>
                <div class="rsys-planet rsys-planet-3"></div>
                
                <div class="rsys-container">
                    <header class="rsys-header">
                        <div class="rsys-logo">
                            <div class="rsys-logo-icon">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="10"/><path d="M12 2a14.5 14.5 0 0 0 0 20 14.5 14.5 0 0 0 0-20"/><path d="M2 12h20"/></svg>
                            </div>
                            <div class="rsys-logo-text">RSYS <span>Cloud</span></div>
                        </div>
                        <div class="rsys-version">v<?php echo RSYS_CDN_VERSION; ?></div>
                    </header>
                    
                    <div class="rsys-status <?php echo $this->options['enabled'] ? 'active' : 'inactive'; ?>">
                        <div class="rsys-status-icon">
                            <?php if ($this->options['enabled']): ?>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>
                            <?php else: ?>
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>
                            <?php endif; ?>
                        </div>
                        <span class="rsys-status-text">
                            <?php echo $this->options['enabled'] ? 'CDN Aktif - İçerikler global ağdan sunuluyor' : 'CDN Pasif - Etkinleştirmek için ayarları yapın'; ?>
                        </span>
                    </div>
                    
                    <div class="rsys-card">
                        <div class="rsys-card-header">
                            <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="3"/><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"/></svg>
                            <h2>CDN Ayarları</h2>
                        </div>
                        <div class="rsys-card-body">
                            <form id="rsys-form">
                                <?php wp_nonce_field('rsys_ajax', '_wpnonce'); ?>
                                
                                <div class="rsys-form-group">
                                    <label class="rsys-toggle">
                                        <input type="checkbox" name="enabled" <?php checked($this->options['enabled']); ?>>
                                        <span class="rsys-toggle-switch"></span>
                                        <span class="rsys-toggle-label">CDN'i Etkinleştir</span>
                                    </label>
                                </div>
                                
                                <div class="rsys-form-group">
                                    <label>CDN URL</label>
                                    <input type="url" name="cdn_url" class="rsys-input" value="<?php echo esc_attr($this->options['cdn_url']); ?>" placeholder="https://siteniz.rsys.cloud">
                                    <p class="rsys-help">RSYS Cloud panelinizdeki CDN URL'nizi buraya girin</p>
                                </div>
                                
                                <div class="rsys-form-group">
                                    <label>CDN'e Dahil Edilecek Dosyalar</label>
                                    <div class="rsys-checkbox-grid">
                                        <label class="rsys-checkbox-item">
                                            <input type="checkbox" name="include_images" <?php checked($this->options['include_images']); ?>>
                                            <span class="rsys-checkbox-box"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><polyline points="20 6 9 17 4 12"/></svg></span>
                                            <span class="rsys-checkbox-label"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"/><circle cx="8.5" cy="8.5" r="1.5"/><polyline points="21 15 16 10 5 21"/></svg> Görseller</span>
                                        </label>
                                        <label class="rsys-checkbox-item">
                                            <input type="checkbox" name="include_css" <?php checked($this->options['include_css']); ?>>
                                            <span class="rsys-checkbox-box"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><polyline points="20 6 9 17 4 12"/></svg></span>
                                            <span class="rsys-checkbox-label"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M4 2v20l2-1 2 1 2-1 2 1 2-1 2 1 2-1 2 1V2l-2 1-2-1-2 1-2-1-2 1-2-1-2 1-2-1z"/></svg> CSS</span>
                                        </label>
                                        <label class="rsys-checkbox-item">
                                            <input type="checkbox" name="include_js" <?php checked($this->options['include_js']); ?>>
                                            <span class="rsys-checkbox-box"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><polyline points="20 6 9 17 4 12"/></svg></span>
                                            <span class="rsys-checkbox-label"><svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"/></svg> JavaScript</span>
                                        </label>
                                    </div>
                                </div>
                                
                                <div class="rsys-form-group">
                                    <label>Hariç Tutulacak Yollar</label>
                                    <input type="text" name="exclude_paths" class="rsys-input" value="<?php echo esc_attr($this->options['exclude_paths']); ?>" placeholder="/wp-admin/, /wp-includes/js/tinymce/">
                                    <p class="rsys-help">CDN'den hariç tutulacak yolları virgülle ayırın</p>
                                </div>
                                
                                <div class="rsys-btn-group">
                                    <button type="submit" class="rsys-btn rsys-btn-primary">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/><polyline points="17 21 17 13 7 13 7 21"/><polyline points="7 3 7 8 15 8"/></svg>
                                        Ayarları Kaydet
                                    </button>
                                    <button type="button" class="rsys-btn rsys-btn-secondary" onclick="testConnection()">
                                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71"/><path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71"/></svg>
                                        Bağlantıyı Test Et
                                    </button>
                                </div>
                            </form>
                            
                            <div class="rsys-info">
                                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="12" y1="16" x2="12" y2="12"/><line x1="12" y1="8" x2="12.01" y2="8"/></svg>
                                <div class="rsys-info-text">
                                    <strong>Henüz hesabınız yok mu?</strong><br>
                                    <a href="<?php echo RSYS_SITE_URL; ?>/register.php" target="_blank">RSYS Cloud'a ücretsiz kaydolun</a> ve her ay 5GB ücretsiz CDN kullanın.
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <footer class="rsys-footer">
                        RSYS Cloud CDN © <?php echo date('Y'); ?> · <a href="<?php echo RSYS_SITE_URL; ?>" target="_blank">rsys.cloud</a>
                    </footer>
                </div>
                
                <div id="rsys-toast" class="rsys-toast"></div>
            </div>
            
            <script>
            (function() {
                const form = document.getElementById('rsys-form');
                const toast = document.getElementById('rsys-toast');
                
                function showToast(message, type) {
                    toast.innerHTML = (type === 'success' ? '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/><polyline points="22 4 12 14.01 9 11.01"/></svg>' : '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"/><line x1="15" y1="9" x2="9" y2="15"/><line x1="9" y1="9" x2="15" y2="15"/></svg>') + message;
                    toast.className = 'rsys-toast ' + type + ' show';
                    setTimeout(() => toast.classList.remove('show'), 3000);
                }
                
                form.addEventListener('submit', async function(e) {
                    e.preventDefault();
                    const btn = form.querySelector('button[type="submit"]');
                    const original = btn.innerHTML;
                    btn.innerHTML = '<span class="rsys-loading"></span> Kaydediliyor...';
                    btn.disabled = true;
                    
                    try {
                        const formData = new FormData(form);
                        formData.append('action', 'rsys_save_settings');
                        const response = await fetch(ajaxurl, { method: 'POST', body: formData });
                        const data = await response.json();
                        
                        if (data.success) {
                            showToast('Ayarlar kaydedildi!', 'success');
                            setTimeout(() => location.reload(), 1000);
                        } else {
                            showToast(data.data || 'Hata oluştu', 'error');
                        }
                    } catch (e) {
                        showToast('Bağlantı hatası', 'error');
                    }
                    btn.innerHTML = original;
                    btn.disabled = false;
                });
                
                window.testConnection = async function() {
                    const url = form.querySelector('input[name="cdn_url"]').value;
                    if (!url) { showToast('CDN URL girin', 'error'); return; }
                    showToast('Test ediliyor...', 'success');
                    try {
                        await fetch(url, { method: 'HEAD', mode: 'no-cors' });
                        showToast('Bağlantı başarılı!', 'success');
                    } catch (e) {
                        showToast('Bağlantı hatası', 'error');
                    }
                };
            })();
            </script>
        </body>
        </html>
        <?php
    }
    
    public function ajax_save_settings() {
        check_ajax_referer('rsys_ajax');
        if (!current_user_can('manage_options')) wp_send_json_error('Yetkisiz');
        
        $this->options = array(
            'enabled' => isset($_POST['enabled']),
            'cdn_url' => esc_url_raw($_POST['cdn_url'] ?? ''),
            'include_images' => isset($_POST['include_images']),
            'include_css' => isset($_POST['include_css']),
            'include_js' => isset($_POST['include_js']),
            'exclude_paths' => sanitize_text_field($_POST['exclude_paths'] ?? '')
        );
        update_option('rsys_cdn_options', $this->options);
        wp_send_json_success();
    }
    
    public function ajax_test_connection() {
        check_ajax_referer('rsys_ajax');
        $url = esc_url_raw($_POST['cdn_url'] ?? '');
        if (empty($url)) wp_send_json_error('CDN URL gerekli');
        $r = wp_remote_head($url, array('timeout' => 10));
        if (is_wp_error($r)) wp_send_json_error($r->get_error_message());
        wp_send_json_success();
    }
    
    public function start_buffer() { ob_start(array($this, 'rewrite_urls')); }
    public function end_buffer() { if (ob_get_level() > 0) ob_end_flush(); }
    
    public function rewrite_urls($content) {
        if (is_admin()) return $content;
        $site = home_url();
        $cdn = rtrim($this->options['cdn_url'], '/');
        if (empty($cdn)) return $content;
        
        $exclude = array_filter(array_map('trim', explode(',', $this->options['exclude_paths'])));
        $ext = array();
        if ($this->options['include_images']) $ext = array_merge($ext, array('jpg','jpeg','png','gif','webp','svg','ico'));
        if ($this->options['include_css']) $ext[] = 'css';
        if ($this->options['include_js']) $ext[] = 'js';
        if (empty($ext)) return $content;
        
        $pattern = '#(?<=["\'\(])' . preg_quote($site, '#') . '(/[^"\')\s]+\.(' . implode('|', $ext) . '))(?=["\'\)])#i';
        return preg_replace_callback($pattern, function($m) use ($cdn, $exclude) {
            foreach ($exclude as $ex) if (strpos($m[1], $ex) !== false) return $m[0];
            return $cdn . $m[1];
        }, $content);
    }
}

RSYS_CDN::get_instance();
